/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.activemq.artemis.utils;

import java.util.function.Supplier;

public abstract class LazyHashProcessor implements HashProcessor {

   private Supplier<HashProcessor> hashProcessorSupplier = Suppliers.memoize(() -> {
      try {
         return createHashProcessor();
      } catch (RuntimeException e) {
         throw e;
      } catch (Exception e) {
         throw new RuntimeException(e);
      }
   });

   @Override
   public String hash(String plainText) throws Exception {
      return hashProcessorSupplier.get().hash(plainText);
   }

   @Override
   public boolean compare(char[] inputValue, String storedHash) {
      return hashProcessorSupplier.get().compare(inputValue, storedHash);
   }

   protected abstract HashProcessor createHashProcessor() throws Exception;
}
