/*
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 */
package org.apache.rat.analysis;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import org.apache.rat.configuration.builders.AnyBuilder;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import static org.assertj.core.api.Assertions.assertThat;

public class GeneratedFileTest {
    private static final String[][] targets = {{"Cayenne", "generated by Cayenne"}, {"JJTree", "Generated By:JJTree"},
            {"JavaCC", "Generated By:JavaCC"}, {"AUTOMATIC", "THIS FILE IS AUTOMATICALLY GENERATED"},
            {"XBeans", "NOTE: this file is autogenerated by XBeans"},
            {"automatic", "This file was automatically generated by"},
            {"do not edit", "# WARNING: DO NOT EDIT OR DELETE THIS WORKSPACE FILE!"},
            {"NMAKE", "# Microsoft Developer Studio Generated NMAKE File"},
            {"MDS Build", "# Microsoft Developer Studio Generated Build File"},
            {"autoheader", "Generated from configure.ac by autoheader"},
            {"aclocal", "generated automatically by aclocal"},
            {"maven project", "build.xml generated by maven from project.xml"},
            {"generated", "This file was generated by"}, {"auto", "This file has been automatically generated."},
            {"do not modify", "Automatically generated - do not modify!"}, {"javadoc style", "Javadoc style sheet"},
            {"SOURCE", "SOURCE FILE GENERATED"}, {"Batik", "Generated by the Batik"},
            {"file autogenerated", "this file is autogenerated"},
            {"class autogenerated", "This class was autogenerated"}, {"maven", "Generated by Maven"},
            {"Thrift", "Autogenerated by Thrift"},
            {"machine generated", "DO NOT EDIT THIS FILE - it is machine generated"},
            {"generated by", "This class was generated by"}, {"javadoc", "Generated by javadoc"}};

    public static Stream<Arguments> parameterProvider() {
        List<Arguments> lst = new ArrayList<>();

        for (String[] target : targets) {
            lst.add(Arguments.of(target[0], target[1]));
        }
        return lst.stream();
    }

    private final IHeaderMatcher matcher;

    GeneratedFileTest() {
        matcher = new AnyBuilder().setResource("/org/apache/rat/generation-keywords.txt").build();
    }

    @ParameterizedTest
    @MethodSource("parameterProvider")
    public void testMatchProcessing(String name, String text) throws IOException {
        assertThat(processText(text)).as(name).isTrue();
    }

    private boolean processText(String text) throws IOException {
        try (BufferedReader in = new BufferedReader(new StringReader(text))) {
            IHeaders headers = HeaderCheckWorker.readHeader(in,
                    HeaderCheckWorker.DEFAULT_NUMBER_OF_RETAINED_HEADER_LINES);
            return matcher.matches(headers);
        }
    }
}
