/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.shaded.org.apache.ignite.internal.binarytuple;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.util.UUID;
import java.util.function.Function;
import org.apache.ignite.shaded.org.apache.ignite.internal.binarytuple.BinaryTupleFormatException;
import org.apache.ignite.shaded.org.apache.ignite.internal.binarytuple.ByteBufferAccessor;
import org.apache.ignite.shaded.org.apache.ignite.internal.util.ByteUtils;

public class BinaryTupleParser {
    public static final ByteOrder ORDER = ByteOrder.LITTLE_ENDIAN;
    private static final int UUID_SIZE = 16;
    private final int numElements;
    private final int entrySize;
    private final int entryBase;
    private final int valueBase;
    protected final ByteBufferAccessor byteBufferAccessor;
    private final OffsetTableReader offsetTableReader;
    protected final ByteBuffer buffer;

    public BinaryTupleParser(int numElements, ByteBuffer buffer) {
        this(numElements, buffer, PlainByteBufferAccessor::new);
    }

    public BinaryTupleParser(int numElements, ByteBuffer buffer, Function<ByteBuffer, ByteBufferAccessor> byteBufferAccessorFactory) {
        this.numElements = numElements;
        assert (buffer.order() == ORDER) : "Buffer order must be LITTLE_ENDIAN, actual: " + String.valueOf(buffer.order());
        assert (buffer.position() == 0) : "Buffer position must be 0, actual: " + buffer.position();
        this.buffer = buffer;
        this.byteBufferAccessor = byteBufferAccessorFactory.apply(buffer);
        byte flags = this.byteBufferAccessor.get(0);
        this.entryBase = 1;
        this.entrySize = 1 << (flags & 3);
        this.valueBase = this.entryBase + this.entrySize * numElements;
        this.offsetTableReader = OffsetTableReader.fromEntrySize(this.entrySize);
    }

    public int size() {
        return this.valueBase + this.getOffset(this.valueBase - this.entrySize);
    }

    public int elementCount() {
        return this.numElements;
    }

    public ByteBuffer byteBuffer() {
        return this.buffer.slice().order(ORDER);
    }

    public ByteBufferAccessor accessor() {
        return this.byteBufferAccessor;
    }

    public void fetch(int index, Sink sink) {
        int nextOffset;
        assert (index >= 0);
        assert (index < this.numElements) : "Index out of bounds: " + index + " >= " + this.numElements;
        int entry = this.entryBase + index * this.entrySize;
        int offset = this.valueBase;
        if (index > 0) {
            offset += this.getOffset(entry - this.entrySize);
        }
        if ((nextOffset = this.valueBase + this.getOffset(entry)) < offset) {
            throw new BinaryTupleFormatException("Corrupted offset table");
        }
        sink.nextElement(index, offset, nextOffset);
    }

    public Readability valueReadability(int index) {
        int nextOffset;
        assert (index >= 0);
        assert (index < this.numElements) : "Index out of bounds: " + index + " >= " + this.numElements;
        int entry = this.entryBase + index * this.entrySize;
        if (entry >= this.buffer.capacity()) {
            return Readability.NOT_READABLE;
        }
        int offset = this.valueBase;
        if (index > 0) {
            offset += this.getOffset(entry - this.entrySize);
        }
        if (offset == (nextOffset = this.valueBase + this.getOffset(entry))) {
            return Readability.READABLE;
        }
        if (offset >= this.buffer.capacity()) {
            return Readability.NOT_READABLE;
        }
        if (nextOffset > this.buffer.capacity()) {
            return Readability.PARTIAL_READABLE;
        }
        return Readability.READABLE;
    }

    public void parse(Sink sink) {
        int entry = this.entryBase;
        int offset = this.valueBase;
        for (int i = 0; i < this.numElements; ++i) {
            int nextOffset = this.valueBase + this.getOffset(entry);
            if (nextOffset < offset) {
                throw new BinaryTupleFormatException("Corrupted offset table");
            }
            sink.nextElement(i, offset, nextOffset);
            offset = nextOffset;
            entry += this.entrySize;
        }
    }

    public static boolean booleanValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        if (len == 1) {
            return ByteUtils.byteToBoolean(byteBufferAccessor.get(begin));
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public static byte byteValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return byteBufferAccessor.get(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public static short shortValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return byteBufferAccessor.get(begin);
            }
            case 2: {
                return byteBufferAccessor.getShort(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public static int intValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return byteBufferAccessor.get(begin);
            }
            case 2: {
                return byteBufferAccessor.getShort(begin);
            }
            case 4: {
                return byteBufferAccessor.getInt(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public static long longValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 1: {
                return byteBufferAccessor.get(begin);
            }
            case 2: {
                return byteBufferAccessor.getShort(begin);
            }
            case 4: {
                return byteBufferAccessor.getInt(begin);
            }
            case 8: {
                return byteBufferAccessor.getLong(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public static float floatValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 4: {
                return byteBufferAccessor.getFloat(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public static double doubleValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 4: {
                return byteBufferAccessor.getFloat(begin);
            }
            case 8: {
                return byteBufferAccessor.getDouble(begin);
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    public BigInteger numberValue(int begin, int end) {
        byte[] bytes;
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.buffer.hasArray()) {
            bytes = this.buffer.array();
            begin += this.buffer.arrayOffset();
        } else {
            bytes = this.getBytes(begin, end);
            begin = 0;
        }
        return new BigInteger(bytes, begin, len);
    }

    public final String stringValue(int begin, int end) {
        byte[] bytes;
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.byteBufferAccessor.get(begin) == -128) {
            ++begin;
            --len;
        }
        if (this.buffer.hasArray()) {
            bytes = this.buffer.array();
            begin += this.buffer.arrayOffset();
        } else {
            bytes = this.getBytes(begin, end);
            begin = 0;
        }
        return new String(bytes, begin, len, StandardCharsets.UTF_8);
    }

    public final byte[] bytesValue(int begin, int end) {
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.byteBufferAccessor.get(begin) == -128) {
            ++begin;
        }
        return this.getBytes(begin, end);
    }

    public final ByteBuffer bytesValueAsBuffer(int begin, int end) {
        int len = end - begin;
        if (len <= 0) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        if (this.byteBufferAccessor.get(begin) == -128) {
            ++begin;
        }
        return this.buffer.duplicate().position(begin).limit(end).slice();
    }

    public final UUID uuidValue(int begin, int end) {
        int len = end - begin;
        if (len != 16) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        long msb = this.byteBufferAccessor.getLong(begin);
        long lsb = this.byteBufferAccessor.getLong(begin + 8);
        return new UUID(msb, lsb);
    }

    public static LocalDate dateValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        if (len != 3) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        return BinaryTupleParser.getDate(byteBufferAccessor, begin);
    }

    public static LocalTime timeValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        if (len < 4 || len > 6) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        return BinaryTupleParser.getTime(byteBufferAccessor, begin, len);
    }

    public static LocalDateTime dateTimeValue(ByteBufferAccessor byteBufferAccessor, int begin, int end) {
        int len = end - begin;
        if (len < 7 || len > 9) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        return LocalDateTime.of(BinaryTupleParser.getDate(byteBufferAccessor, begin), BinaryTupleParser.getTime(byteBufferAccessor, begin + 3, len - 3));
    }

    public final Instant timestampValue(int begin, int end) {
        int len = end - begin;
        if (len != 8 && len != 12) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        long seconds = this.byteBufferAccessor.getLong(begin);
        int nanos = len == 8 ? 0 : this.byteBufferAccessor.getInt(begin + 8);
        return Instant.ofEpochSecond(seconds, nanos);
    }

    public final Duration durationValue(int begin, int end) {
        int len = end - begin;
        if (len != 8 && len != 12) {
            throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
        }
        long seconds = this.byteBufferAccessor.getLong(begin);
        int nanos = len == 8 ? 0 : this.byteBufferAccessor.getInt(begin + 8);
        return Duration.ofSeconds(seconds, nanos);
    }

    public final Period periodValue(int begin, int end) {
        int len = end - begin;
        switch (len) {
            case 3: {
                return Period.of(this.byteBufferAccessor.get(begin), this.byteBufferAccessor.get(begin + 1), this.byteBufferAccessor.get(begin + 2));
            }
            case 6: {
                return Period.of(this.byteBufferAccessor.getShort(begin), this.byteBufferAccessor.getShort(begin + 2), this.byteBufferAccessor.getShort(begin + 4));
            }
            case 12: {
                return Period.of(this.byteBufferAccessor.getInt(begin), this.byteBufferAccessor.getInt(begin + 4), this.byteBufferAccessor.getInt(begin + 8));
            }
        }
        throw new BinaryTupleFormatException("Invalid length for a tuple element: " + len);
    }

    private int getOffset(int index) {
        return this.offsetTableReader.read(this.byteBufferAccessor, index);
    }

    private byte[] getBytes(int begin, int end) {
        byte[] bytes = new byte[end - begin];
        this.buffer.duplicate().position(begin).limit(end).get(bytes);
        return bytes;
    }

    private static LocalDate getDate(ByteBufferAccessor byteBufferAccessor, int offset) {
        int date = Short.toUnsignedInt(byteBufferAccessor.getShort(offset));
        int day = (date |= byteBufferAccessor.get(offset + 2) << 16) & 0x1F;
        int month = date >> 5 & 0xF;
        int year = date >> 9;
        return LocalDate.of(year, month, day);
    }

    private static LocalTime getTime(ByteBufferAccessor byteBufferAccessor, int offset, int length) {
        int nanos;
        long time = Integer.toUnsignedLong(byteBufferAccessor.getInt(offset));
        switch (length) {
            case 4: {
                nanos = ((int)time & 0x3FF) * 1000 * 1000;
                time >>>= 10;
                break;
            }
            case 5: {
                nanos = ((int)(time |= Byte.toUnsignedLong(byteBufferAccessor.get(offset + 4)) << 32) & 0xFFFFF) * 1000;
                time >>>= 20;
                break;
            }
            case 6: {
                nanos = (int)(time |= Short.toUnsignedLong(byteBufferAccessor.getShort(offset + 4)) << 32) & 0x3FFFFFFF;
                time >>>= 30;
                break;
            }
            default: {
                throw new BinaryTupleFormatException("Invalid length for a tuple element: " + length);
            }
        }
        int second = (int)time & 0x3F;
        int minute = (int)time >>> 6 & 0x3F;
        int hour = (int)time >>> 12 & 0x1F;
        return LocalTime.of(hour, minute, second, nanos);
    }

    private static enum OffsetTableReader {
        BYTE_ENTRY{

            @Override
            int read(ByteBufferAccessor bufferAccessor, int index) {
                return Byte.toUnsignedInt(bufferAccessor.get(index));
            }
        }
        ,
        SHORT_ENTRY{

            @Override
            int read(ByteBufferAccessor bufferAccessor, int index) {
                return Short.toUnsignedInt(bufferAccessor.getShort(index));
            }
        }
        ,
        INTEGER_ENTRY{

            @Override
            int read(ByteBufferAccessor bufferAccessor, int index) {
                int offset = bufferAccessor.getInt(index);
                if (offset < 0) {
                    throw new BinaryTupleFormatException("Unsupported offset table size");
                }
                return offset;
            }
        };


        static OffsetTableReader fromEntrySize(int size) {
            switch (size) {
                case 1: {
                    return BYTE_ENTRY;
                }
                case 2: {
                    return SHORT_ENTRY;
                }
                case 4: {
                    return INTEGER_ENTRY;
                }
                case 8: {
                    throw new BinaryTupleFormatException("Unsupported offset table size");
                }
            }
            throw new BinaryTupleFormatException("Invalid offset table size");
        }

        abstract int read(ByteBufferAccessor var1, int var2);
    }

    public static interface Sink {
        public void nextElement(int var1, int var2, int var3);
    }

    public static enum Readability {
        NOT_READABLE,
        READABLE,
        PARTIAL_READABLE;

    }

    private static class PlainByteBufferAccessor
    implements ByteBufferAccessor {
        private final ByteBuffer buffer;

        PlainByteBufferAccessor(ByteBuffer buffer) {
            this.buffer = buffer;
        }

        @Override
        public byte get(int index) {
            return this.buffer.get(index);
        }

        @Override
        public short getShort(int index) {
            return this.buffer.getShort(index);
        }

        @Override
        public int getInt(int index) {
            return this.buffer.getInt(index);
        }

        @Override
        public long getLong(int index) {
            return this.buffer.getLong(index);
        }

        @Override
        public float getFloat(int index) {
            return this.buffer.getFloat(index);
        }

        @Override
        public double getDouble(int index) {
            return this.buffer.getDouble(index);
        }

        @Override
        public int capacity() {
            return this.buffer.capacity();
        }
    }
}

