/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql.catalyst.plans.logical

import org.apache.spark.SparkException
import org.apache.spark.sql.AnalysisException
import org.apache.spark.sql.catalyst.expressions.{AnalysisAwareExpression, Expression, Literal, UnaryExpression, Unevaluable}
import org.apache.spark.sql.catalyst.parser.ParserInterface
import org.apache.spark.sql.catalyst.trees.TreePattern.{ANALYSIS_AWARE_EXPRESSION, TreePattern}
import org.apache.spark.sql.catalyst.util.{CharVarcharUtils, GeneratedColumn, IdentityColumn, V2ExpressionBuilder}
import org.apache.spark.sql.catalyst.util.ResolveDefaultColumns.validateDefaultValueExpr
import org.apache.spark.sql.catalyst.util.ResolveDefaultColumnsUtils.{CURRENT_DEFAULT_COLUMN_METADATA_KEY, EXISTS_DEFAULT_COLUMN_METADATA_KEY}
import org.apache.spark.sql.connector.catalog.{Column => V2Column, ColumnDefaultValue, DefaultValue, IdentityColumnSpec}
import org.apache.spark.sql.connector.catalog.CatalogV2Implicits.MultipartIdentifierHelper
import org.apache.spark.sql.connector.expressions.LiteralValue
import org.apache.spark.sql.errors.QueryCompilationErrors
import org.apache.spark.sql.internal.connector.ColumnImpl
import org.apache.spark.sql.types.{DataType, Metadata, MetadataBuilder, StructField}

/**
 * User-specified column definition for CREATE/REPLACE TABLE commands. This is an expression so that
 * analyzer can resolve the default value expression automatically.
 *
 * For CREATE/REPLACE TABLE commands, columns are created from scratch, so we store the
 * user-specified default value as both the current default and exists default, in methods
 * `toV1Column` and `toV2Column`.
 */
case class ColumnDefinition(
    name: String,
    dataType: DataType,
    nullable: Boolean = true,
    comment: Option[String] = None,
    defaultValue: Option[DefaultValueExpression] = None,
    generationExpression: Option[String] = None,
    identityColumnSpec: Option[IdentityColumnSpec] = None,
    metadata: Metadata = Metadata.empty) extends Expression with Unevaluable {
  assert(
    generationExpression.isEmpty || identityColumnSpec.isEmpty,
    "A ColumnDefinition cannot contain both a generation expression and an identity column spec.")

  override def children: Seq[Expression] = defaultValue.toSeq

  override protected def withNewChildrenInternal(
      newChildren: IndexedSeq[Expression]): Expression = {
    copy(defaultValue = newChildren.headOption.map(_.asInstanceOf[DefaultValueExpression]))
  }

  def toV2Column(statement: String): V2Column = {
    ColumnImpl(
      name,
      dataType,
      nullable,
      comment.orNull,
      defaultValue.map(_.toV2(statement, name)).orNull,
      generationExpression.orNull,
      identityColumnSpec.orNull,
      if (metadata == Metadata.empty) null else metadata.json)
  }

  def toV1Column: StructField = {
    val metadataBuilder = new MetadataBuilder().withMetadata(metadata)
    comment.foreach { c =>
      metadataBuilder.putString("comment", c)
    }
    defaultValue.foreach { default =>
      metadataBuilder.putExpression(
        CURRENT_DEFAULT_COLUMN_METADATA_KEY, default.originalSQL, Some(default.child))
      val existsSQL = default.child match {
        case l: Literal => l.sql
        case _ => default.originalSQL
      }
      metadataBuilder.putString(EXISTS_DEFAULT_COLUMN_METADATA_KEY, existsSQL)
    }
    generationExpression.foreach { generationExpr =>
      metadataBuilder.putString(GeneratedColumn.GENERATION_EXPRESSION_METADATA_KEY, generationExpr)
    }
    encodeIdentityColumnSpec(metadataBuilder)
    StructField(name, dataType, nullable, metadataBuilder.build())
  }

  private def encodeIdentityColumnSpec(metadataBuilder: MetadataBuilder): Unit = {
    identityColumnSpec.foreach { spec: IdentityColumnSpec =>
      metadataBuilder.putLong(IdentityColumn.IDENTITY_INFO_START, spec.getStart)
      metadataBuilder.putLong(IdentityColumn.IDENTITY_INFO_STEP, spec.getStep)
      metadataBuilder.putBoolean(
        IdentityColumn.IDENTITY_INFO_ALLOW_EXPLICIT_INSERT,
        spec.isAllowExplicitInsert)
    }
  }

  /**
   * Returns true if the default value's type has been coerced to match this column's dataType.
   * After type coercion, the default value expression's dataType should match the column's
   * dataType (with CHAR/VARCHAR replaced by STRING).
   */
  def isDefaultValueTypeCoerced: Boolean = defaultValue.forall { d =>
    ColumnDefinition.isDefaultValueTypeMatched(d.child.dataType, dataType)
  }
}

object ColumnDefinition {

  /**
   * Returns true if the default value's type matches the target column type.
   * CHAR/VARCHAR types are replaced with STRING before comparison since type coercion
   * converts them to STRING.
   */
  def isDefaultValueTypeMatched(defaultValueType: DataType, targetType: DataType): Boolean = {
    val expectedType = CharVarcharUtils.replaceCharVarcharWithString(targetType)
    defaultValueType == expectedType
  }

  def fromV1Column(col: StructField, parser: ParserInterface): ColumnDefinition = {
    val metadataBuilder = new MetadataBuilder().withMetadata(col.metadata)
    metadataBuilder.remove("comment")
    metadataBuilder.remove(CURRENT_DEFAULT_COLUMN_METADATA_KEY)
    metadataBuilder.remove(EXISTS_DEFAULT_COLUMN_METADATA_KEY)
    metadataBuilder.remove(GeneratedColumn.GENERATION_EXPRESSION_METADATA_KEY)
    metadataBuilder.remove(IdentityColumn.IDENTITY_INFO_START)
    metadataBuilder.remove(IdentityColumn.IDENTITY_INFO_STEP)
    metadataBuilder.remove(IdentityColumn.IDENTITY_INFO_ALLOW_EXPLICIT_INSERT)

    val hasDefaultValue = col.getCurrentDefaultValue().isDefined &&
      col.getExistenceDefaultValue().isDefined
    val defaultValue = if (hasDefaultValue) {
      // `ColumnDefinition` is for CREATE/REPLACE TABLE commands, and it only needs one
      // default value. Here we assume user wants the current default of the v1 column to be
      // the default value of this column definition.
      val defaultValueSQL = col.getCurrentDefaultValue().get
      Some(DefaultValueExpression(parser.parseExpression(defaultValueSQL), defaultValueSQL))
    } else {
      None
    }
    val generationExpr = GeneratedColumn.getGenerationExpression(col)
    val identityColumnSpec = if (col.metadata.contains(IdentityColumn.IDENTITY_INFO_START)) {
      Some(new IdentityColumnSpec(
        col.metadata.getLong(IdentityColumn.IDENTITY_INFO_START),
        col.metadata.getLong(IdentityColumn.IDENTITY_INFO_STEP),
        col.metadata.getBoolean(IdentityColumn.IDENTITY_INFO_ALLOW_EXPLICIT_INSERT)
      ))
    } else {
      None
    }
    ColumnDefinition(
      col.name,
      col.dataType,
      col.nullable,
      col.getComment(),
      defaultValue,
      generationExpr,
      identityColumnSpec,
      metadataBuilder.build()
    )
  }

  // Called by `CheckAnalysis` to check column definitions in DDL commands.
  def checkColumnDefinitions(plan: LogicalPlan): Unit = {
    plan match {
      // Do not check anything if the children are not resolved yet.
      case _ if !plan.childrenResolved =>

      // Wrap errors for default values in a more user-friendly message.
      case cmd: V2CreateTablePlan if cmd.columns.exists(_.defaultValue.isDefined) =>
        val statement = cmd match {
          case _: CreateTable => "CREATE TABLE"
          case _: ReplaceTable => "REPLACE TABLE"
          case other =>
            val cmd = other.getClass.getSimpleName
            throw SparkException.internalError(
              s"Command $cmd should not have column default value expression.")
        }
        cmd.columns.foreach { col =>
          col.defaultValue.foreach { default =>
            checkDefaultColumnConflicts(col)
            validateDefaultValueExpr(default, statement, col.name, Some(col.dataType))
          }
        }

      case cmd: AddColumns if cmd.columnsToAdd.exists(_.default.isDefined) =>
        cmd.columnsToAdd.foreach { c =>
          c.default.foreach { d =>
            validateDefaultValueExpr(d, "ALTER TABLE ADD COLUMNS", c.colName, Some(c.dataType))
          }
        }

      case cmd: AlterColumns if cmd.specs.exists(_.newDefaultExpression.isDefined) =>
        cmd.specs.foreach { c =>
          c.newDefaultExpression.foreach { d =>
            validateDefaultValueExpr(d, "ALTER TABLE ALTER COLUMN", c.column.name.quoted,
              None)
          }
        }

      case _ =>
    }
  }

  private def checkDefaultColumnConflicts(col: ColumnDefinition): Unit = {
    if (col.generationExpression.isDefined) {
      throw new AnalysisException(
        errorClass = "GENERATED_COLUMN_WITH_DEFAULT_VALUE",
        messageParameters = Map(
          "colName" -> col.name,
          "defaultValue" -> col.defaultValue.get.originalSQL,
          "genExpr" -> col.generationExpression.get
        )
      )
    }
    if (col.identityColumnSpec.isDefined) {
      throw new AnalysisException(
        errorClass = "IDENTITY_COLUMN_WITH_DEFAULT_VALUE",
        messageParameters = Map(
          "colName" -> col.name,
          "defaultValue" -> col.defaultValue.get.originalSQL,
          "identityColumnSpec" -> col.identityColumnSpec.get.toString
        )
      )
    }
  }
}

/**
 * A fake expression to hold the column/variable default value expression and its original SQL text.
 */
case class DefaultValueExpression(
    child: Expression,
    originalSQL: String,
    analyzedChild: Option[Expression] = None)
  extends UnaryExpression
  with Unevaluable
  with AnalysisAwareExpression[DefaultValueExpression] {

  final override val nodePatterns: Seq[TreePattern] = Seq(ANALYSIS_AWARE_EXPRESSION)

  override def dataType: DataType = child.dataType
  override def stringArgs: Iterator[Any] = Iterator(child, originalSQL)
  override def markAsAnalyzed(): DefaultValueExpression =
    copy(analyzedChild = Some(child))
  override protected def withNewChildInternal(newChild: Expression): Expression =
    copy(child = newChild)

  // Convert the default expression to ColumnDefaultValue, which is required by DS v2 APIs.
  def toV2(statement: String, colName: String): ColumnDefaultValue = child match {
    case Literal(value, dataType) =>
      val currentDefault = analyzedChild.flatMap(new V2ExpressionBuilder(_).build())
      val existsDefault = LiteralValue(value, dataType)
      new ColumnDefaultValue(originalSQL, currentDefault.orNull, existsDefault)
    case _ =>
      throw QueryCompilationErrors.defaultValueNotConstantError(statement, colName, originalSQL)
  }

  // Convert the default expression to DefaultValue, which is required by DS v2 APIs.
  def toV2CurrentDefault(statement: String, colName: String): DefaultValue = child match {
    case Literal(_, _) =>
      val currentDefault = analyzedChild.flatMap(new V2ExpressionBuilder(_).build())
      new DefaultValue(originalSQL, currentDefault.orNull)
    case _ =>
      throw QueryCompilationErrors.defaultValueNotConstantError(statement, colName, originalSQL)
  }
}
