"use strict";
/*
Copyright 2019 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.IrcRoom = void 0;
exports.validateChannelName = validateChannelName;
const matrix_appservice_bridge_1 = require("matrix-appservice-bridge");
const formatting_1 = require("../irc/formatting");
/**
 * Validate that a channel name is in the correct form.
 * NOTE: This accepts both # and & as channel prefixes.
 * @param channel Channel name
 * @returns True if the channel is valid, or false otherwise.
 */
function validateChannelName(channel) {
    // According to https://www.rfc-editor.org/rfc/rfc1459#section-1.3
    // eslint-disable-next-line no-control-regex
    return /^#([^:\x00-\x1F\s,]){1,199}$/.test(channel);
}
class IrcRoom extends matrix_appservice_bridge_1.RemoteRoom {
    server;
    channel;
    /**
     * Construct a new IRC room.
     * @constructor
     * @param {IrcServer} server : The IRC server which contains this room.
     * @param {String} channel : The channel this room represents.
     */
    constructor(server, channel) {
        // Because `super` must be called first, we convert the case several times.
        super(IrcRoom.createId(server, (0, formatting_1.toIrcLowerCase)(channel)), {
            domain: server.domain,
            channel: (0, formatting_1.toIrcLowerCase)(channel),
            type: channel.startsWith("#") ? "channel" : "pm"
        });
        this.server = server;
        this.channel = channel;
        if (!server || !channel) {
            throw new Error("Server and channel are required.");
        }
        channel = (0, formatting_1.toIrcLowerCase)(channel);
    }
    getDomain() {
        return super.get("domain");
    }
    getServer() {
        return this.server;
    }
    getChannel() {
        return super.get("channel");
    }
    getType() {
        return super.get("type");
    }
    static fromRemoteRoom(server, remoteRoom) {
        return new IrcRoom(server, remoteRoom.get("channel"));
    }
    // An IRC room is uniquely identified by a combination of the channel name and the
    // IRC network the channel resides on. Space is the delimiter because neither the
    // domain nor the channel allows spaces.
    static createId(server, channel) {
        return server.domain + " " + channel;
    }
}
exports.IrcRoom = IrcRoom;
//# sourceMappingURL=IrcRoom.js.map