﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/NetworkInterfaceAssociation.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes the private IPv4 address of a network interface.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/NetworkInterfacePrivateIpAddress">AWS
 * API Reference</a></p>
 */
class NetworkInterfacePrivateIpAddress {
 public:
  AWS_EC2_API NetworkInterfacePrivateIpAddress() = default;
  AWS_EC2_API NetworkInterfacePrivateIpAddress(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API NetworkInterfacePrivateIpAddress& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The association information for an Elastic IP address (IPv4) associated with
   * the network interface.</p>
   */
  inline const NetworkInterfaceAssociation& GetAssociation() const { return m_association; }
  inline bool AssociationHasBeenSet() const { return m_associationHasBeenSet; }
  template <typename AssociationT = NetworkInterfaceAssociation>
  void SetAssociation(AssociationT&& value) {
    m_associationHasBeenSet = true;
    m_association = std::forward<AssociationT>(value);
  }
  template <typename AssociationT = NetworkInterfaceAssociation>
  NetworkInterfacePrivateIpAddress& WithAssociation(AssociationT&& value) {
    SetAssociation(std::forward<AssociationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether this IPv4 address is the primary private IPv4 address of
   * the network interface.</p>
   */
  inline bool GetPrimary() const { return m_primary; }
  inline bool PrimaryHasBeenSet() const { return m_primaryHasBeenSet; }
  inline void SetPrimary(bool value) {
    m_primaryHasBeenSet = true;
    m_primary = value;
  }
  inline NetworkInterfacePrivateIpAddress& WithPrimary(bool value) {
    SetPrimary(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The private DNS name.</p>
   */
  inline const Aws::String& GetPrivateDnsName() const { return m_privateDnsName; }
  inline bool PrivateDnsNameHasBeenSet() const { return m_privateDnsNameHasBeenSet; }
  template <typename PrivateDnsNameT = Aws::String>
  void SetPrivateDnsName(PrivateDnsNameT&& value) {
    m_privateDnsNameHasBeenSet = true;
    m_privateDnsName = std::forward<PrivateDnsNameT>(value);
  }
  template <typename PrivateDnsNameT = Aws::String>
  NetworkInterfacePrivateIpAddress& WithPrivateDnsName(PrivateDnsNameT&& value) {
    SetPrivateDnsName(std::forward<PrivateDnsNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The private IPv4 address.</p>
   */
  inline const Aws::String& GetPrivateIpAddress() const { return m_privateIpAddress; }
  inline bool PrivateIpAddressHasBeenSet() const { return m_privateIpAddressHasBeenSet; }
  template <typename PrivateIpAddressT = Aws::String>
  void SetPrivateIpAddress(PrivateIpAddressT&& value) {
    m_privateIpAddressHasBeenSet = true;
    m_privateIpAddress = std::forward<PrivateIpAddressT>(value);
  }
  template <typename PrivateIpAddressT = Aws::String>
  NetworkInterfacePrivateIpAddress& WithPrivateIpAddress(PrivateIpAddressT&& value) {
    SetPrivateIpAddress(std::forward<PrivateIpAddressT>(value));
    return *this;
  }
  ///@}
 private:
  NetworkInterfaceAssociation m_association;

  bool m_primary{false};

  Aws::String m_privateDnsName;

  Aws::String m_privateIpAddress;
  bool m_associationHasBeenSet = false;
  bool m_primaryHasBeenSet = false;
  bool m_privateDnsNameHasBeenSet = false;
  bool m_privateIpAddressHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
